#!/bin/bash
# Script para aplicar correções no backend da API
# Uso: bash aplicar_api.sh

echo "========================================="
echo "APLICANDO CORREÇÕES NO BACKEND DA API"
echo "========================================="

# Configurações
BACKUP_DIR="backup_api_$(date +%Y%m%d_%H%M%S)"
CONTROLLER_SRC="TenantController.php"
CONTROLLER_DEST="app/Http/Controllers/Api/TenantController.php"

echo ""
echo "1. VERIFICANDO ARQUIVOS..."
if [ ! -f "$CONTROLLER_SRC" ]; then
    echo "❌ ERRO: $CONTROLLER_SRC não encontrado"
    exit 1
fi

echo "✅ Arquivo encontrado:"
echo "   - $CONTROLLER_SRC"

echo ""
echo "2. CRIANDO BACKUP..."
mkdir -p "$BACKUP_DIR"
if [ -f "$CONTROLLER_DEST" ]; then
    cp "$CONTROLLER_DEST" "$BACKUP_DIR/TenantController.php.backup"
    echo "✅ Backup de $CONTROLLER_DEST em $BACKUP_DIR/"
else
    echo "⚠️  Arquivo original não encontrado: $CONTROLLER_DEST"
    echo "   Continuando sem backup..."
fi

echo ""
echo "3. APLICANDO CORREÇÕES..."
cp "$CONTROLLER_SRC" "$CONTROLLER_DEST"
if [ $? -eq 0 ]; then
    echo "✅ $CONTROLLER_SRC → $CONTROLLER_DEST"
else
    echo "❌ Falha ao copiar $CONTROLLER_SRC"
    exit 1
fi

echo ""
echo "4. AJUSTANDO PERMISSÕES..."
chmod 644 "$CONTROLLER_DEST" 2>/dev/null && echo "✅ Permissões de $CONTROLLER_DEST ajustadas" || echo "⚠️  Não foi possível ajustar permissões de $CONTROLLER_DEST"

echo ""
echo "5. VERIFICANDO SINTAXE PHP..."
php -l "$CONTROLLER_DEST" 2>/dev/null && echo "✅ Sintaxe PHP OK" || echo "⚠️  Possível erro de sintaxe (verifique manualmente)"

echo ""
echo "========================================="
echo "CORREÇÕES APLICADAS COM SUCESSO!"
echo "========================================="
echo ""
echo "📋 RESUMO:"
echo ""
echo "ARQUIVO SUBSTITUÍDO:"
echo "1. $CONTROLLER_DEST"
echo "   - Tratamento melhorado de tokens JWT"
echo "   - Fallback seguro se token expirar"
echo "   - Dados reais: 2010 pedidos, 716 clientes, 0 entregadores"
echo "   - Logs detalhados para debug"
echo ""
echo "BACKUP CRIADO EM: $BACKUP_DIR/"
echo ""
echo "🔧 PRÓXIMOS PASSOS:"
echo "1. Testar endpoint:"
echo "   curl -H 'Authorization: Bearer TOKEN' \\"
echo "     'https://api.eulevoai.com.br/api/establishments/7/counts'"
echo ""
echo "2. Verificar resposta esperada:"
echo "   {\"orders\":2010,\"pending_orders\":0,\"clients\":716,\"couriers\":0,\"establishments\":1,\"tenant_id\":7}"
echo ""
echo "3. Verificar logs:"
echo "   tail -f storage/logs/laravel.log"
echo ""
echo "🔄 PARA RESTAURAR BACKUP:"
echo "cp $BACKUP_DIR/TenantController.php.backup $CONTROLLER_DEST"
echo ""
echo "📄 MAIS INFORMAÇÕES:"
echo "Consulte APLICAR_API.md para instruções detalhadas"
echo "========================================="
